from pyparsing import Literal, Optional, ZeroOrMore, OneOrMore, CharsNotIn

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, LinkEvidence
from codeable_detectors.pyparsing_patterns import ID, qualified_id, round_braces_block


def detect_python_import(ctx, package_name, import_name_pattern=ID):
    # if import_name_pattern is just a specific string, we convert it into the respective Literal
    if isinstance(import_name_pattern, str):
        import_name_pattern = Literal(import_name_pattern)

    python_import_as_name = qualified_id + Optional(Literal("as") + ID)
    python_import_as_names = python_import_as_name + ZeroOrMore(Literal(",") + python_import_as_name)

    matches = []
    import_as_name_matches = ctx.matches_pattern(Literal("import") + Literal(package_name) +
                                                 Optional(Literal("as") + ID))
    for import_as_name_match in import_as_name_matches:
        import_as_name_ctx_elements = import_as_name_match.text.split()
        if len(import_as_name_ctx_elements) == 4:
            package_alias = import_as_name_ctx_elements[3]
        else:
            package_alias = None
        import_as_name_match.update_keyword_args(package_name=package_name,
                                                 package_alias=package_alias, is_package_only_import=True,
                                                 import_name_pattern=import_name_pattern)
        matches.append(import_as_name_match)

    import_from_matches = ctx.matches_pattern(Literal("from") + Literal(package_name) + Literal("import") +
                                              (Literal("*") | round_braces_block | python_import_as_names))
    for import_from_match in import_from_matches:
        imports_string = import_from_match.text[import_from_match.text.find("import") + 6:].strip()
        alias = None
        if imports_string == "*":
            import_from_match.update_keyword_args(package_name=package_name,
                                                  alias=alias, import_name_pattern=import_name_pattern,
                                                  is_package_only_import=False)
            matches.append(import_from_match)
        else:
            if "," in imports_string:
                imports = imports_string.split(",")
            else:
                imports = [imports_string]
            for import_element in imports:
                import_elements = import_element.split()
                name_matches = DetectorContext(import_elements[0]).matches_pattern(import_name_pattern)
                if name_matches:
                    if len(import_elements) == 3 and import_elements[1].strip() == "as":
                        alias = import_elements[2]
                    import_from_match.update_keyword_args(package_name=package_name, alias=alias,
                                                          import_name_pattern=import_name_pattern,
                                                          is_package_only_import=False)
                    matches.append(import_from_match)
    return matches


def get_python_variable_assignments(ctx):
    variable_assignment_pattern = qualified_id + Literal("=") + OneOrMore(CharsNotIn('\n'))
    variable_assignment_pattern.setWhitespaceChars('\n')
    return ctx.matches_pattern(variable_assignment_pattern)


class PythonImportLink(AtLeastOneFileMatchesDetector):
    def __init__(self, package_name):
        super().__init__()
        self.file_endings = ["py"]
        self.package_name = package_name

    def detect_in_context(self, ctx, **kwargs):
        import_matches = detect_python_import(ctx, self.package_name)
        if not import_matches:
            return FailedEvidence("python '" + str(self.package_name) + "' import not found")
        return LinkEvidence(import_matches).set_properties(kwargs=kwargs)


# determine the possible import strings, e.g. for from f import x, x is possible,
# whereas in import f, then f.x is possible.
def get_import_patterns_from_imports_evidences(import_matches):
    import_patterns = []
    for import_match in import_matches:
        package_pre_pattern = None
        if import_match.kwargs["is_package_only_import"]:
            if import_match.kwargs["package_alias"]:
                package_pre_pattern = import_match.kwargs["package_alias"] + "."
            else:
                package_pre_pattern = Literal(import_match.kwargs["package_name"]) + Literal(".")
        if "alias" in import_match.kwargs and import_match.kwargs["alias"]:
            import_pattern = Literal(import_match.kwargs["alias"])
        else:
            import_pattern = import_match.kwargs["import_name_pattern"]
        if package_pre_pattern:
            import_patterns.append(package_pre_pattern + import_pattern)
        else:
            import_patterns.append(import_pattern)
    return import_patterns
